#!/usr/bin/env python
# vim:fileencoding=utf-8
from __future__ import (unicode_literals, division, absolute_import,
                        print_function)

__license__ = 'GPL v3'
__copyright__ = '2014, David Forrester <davidfor@internode.on.net>'

import re
try:
    from PyQt5.Qt import QAction, QInputDialog, QMenu, QIcon
except:
    from PyQt4.Qt import QAction, QInputDialog, QMenu, QIcon

from cssutils.css import CSSRule

from calibre.gui2.tweak_book import NonReplaceDict
# The base class that all tools must inherit from
from calibre.gui2.tweak_book.plugin import Tool

from calibre import force_unicode
from calibre.gui2 import error_dialog
from calibre.ebooks.oeb.polish.container import OEB_DOCS, OEB_STYLES, serialize
from calibre.ebooks.oeb.polish.pretty import pretty_css

from calibre_plugins.sortcss.common_utils import create_menu_action_unique

actions = NonReplaceDict()

class SortCSSTool(Tool):

    #: Set this to a unique name it will be used as a key
    name = 'SortCSS'

    #: If True the user can choose to place this tool in the plugins toolbar
    allowed_in_toolbar = True

    #: If True the user can choose to place this tool in the plugins menu
    allowed_in_menu = False

    def create_action(self, for_toolbar=True):
        # Create an action, this will be added to the plugins toolbar and
        # the plugins menu
        ac = QAction(get_icons('images/icon.png'), 'SortCSS', self.gui)  # noqa
        if not for_toolbar:
            # Register a keyboard shortcut for this toolbar action. We only
            # register it for the action created for the menu, not the toolbar,
            # to avoid a double trigger
            self.register_shortcut(ac, 'sort-css-tool', default_keys=('Ctrl+Shift+Alt+D',))
        self.menu = QMenu(self.gui)
        ac.setMenu(self.menu)
        self.menu.aboutToShow.connect(self.about_to_show_menu)
        ac.triggered.connect(self.sort_stylesheets)
        return ac

    def about_to_show_menu(self):
        self.rebuild_menus()

    def unique_name(self):
        bn = self.name
        return u'Interface Action: %s (%s)'%(bn, self.name)


    def rebuild_menus(self):

        def reg(icon, text, target, sid, keys, description, toolbar_allowed=False):
            if not isinstance(icon, QIcon):
                icon = QIcon(I(icon))
            ac = actions[sid] = QAction(icon, text, self.gui) if icon else QAction(text, self.gui)
            ac.setObjectName('action-' + sid)
#            if toolbar_allowed:
#                toolbar_actions[sid] = ac
            if target is not None:
                ac.triggered.connect(target)
            if isinstance(keys, type('')):
                keys = (keys,)
#            self.keyboard.register_shortcut(
#                sid, unicode(ac.text()).replace('&', ''), default_keys=keys, description=description, action=ac, group=group)
            self.addAction(ac)
            return ac
        
        self.menu.clear()
#        self.actions_unique_map = {}
#
#        self.set_reader_fonts_action = self.create_menu_item_ex(
#                                            self.menu,  _("&Sort CSS"), 
#                                            unique_name='Sort CSS',
#                                            shortcut_name= _("Sort CSS"),
#                                            triggered=self.sort_stylesheets,
#                                            enabled=True
#                                            )

        text = _("&Sort CSS")
        desc = _("Sort CSS")
#        ac = reg(create_icon(name), text, ('rename_block_tag', name), 'rename-block-tag-' + name, 'Ctrl+%d' % (i + 1), desc, syntaxes=())
        ac = reg(None, text, self.sort_stylesheets, 'sort-css', None, desc)
        ac.setToolTip(desc)

        # Before we finalize, make sure we delete any actions for menus that are no longer displayed
#        for menu_id, unique_name in self.old_actions_unique_map.iteritems():
#            if menu_id not in self.actions_unique_map:
#                self.gui.keyboard.unregister_shortcut(unique_name)
#        self.old_actions_unique_map = self.actions_unique_map
#        self.gui.keyboard.finalize()            

    def create_menu_item_ex(self, parent_menu, menu_text, image=None, tooltip=None,
                           shortcut=None, triggered=None, is_checked=None, shortcut_name=None,
                           unique_name=None, enabled=False, is_library_action=False, is_device_action=False):
        ac = create_menu_action_unique(self, parent_menu, menu_text, image, tooltip,
                                       shortcut, triggered, is_checked, shortcut_name, unique_name)
        self.actions_unique_map[ac.calibre_shortcut_unique_name] = ac.calibre_shortcut_unique_name
        ac.setEnabled(enabled)

        if is_library_action:
            self.library_actions_map[shortcut_name] = ac
        if is_device_action:
            self.device_actions_map[shortcut_name] = ac
        return ac


    def sort_stylesheets(self):
        container = self.current_container

        # Ensure any in progress editing the user is doing is present in the container
        self.boss.commit_all_editors_to_container()
        try:
            for name, media_type in container.mime_map.iteritems():
                if media_type in OEB_STYLES:
                    # A stylesheet. Parsed stylesheets are cssutils CSSStylesheet
                    # objects.
                    self.sort_stylesheet(name)
        except Exception:
            # Something bad happened report the error to the user
            import traceback
            error_dialog(self.gui, _('Failed to sort the CSS'), _(
                'Failed to sort the CSS, click "Show details" for more info'),
                det_msg=traceback.format_exc(), show=True)
            # Revert to the saved restore point
            self.boss.revert_requested(self.boss.global_undo.previous_container)
        else:
            # Show the user what changes we have made, allowing her to
            # revert them if necessary
            self.boss.show_current_diff()
            # Update the editor UI to take into account all the changes we
            # have made
            self.boss.apply_container_update_to_gui()

    def sort_stylesheet(self, stylesheet_name):
        # Magnify all font sizes defined in the book by the specified factor
        # First we create a restore point so that the user can undo all changes
        # we make.
        self.boss.add_savepoint('Before: sort_css')

        container = self.current_container  # The book being edited as a container object

        sheet = container.parsed(stylesheet_name)
        rules = sheet.cssRules
#        rules.sort(cmp=lambda x,y:cmp(x.cssText.lower(), y.cssText.lower()))
        rules.sort(cmp=lambda x,y:cmp(x.selectorText.lower(), y.selectorText.lower()))
        container.dirty(stylesheet_name)  # Tell the container that we have changed the stylesheet

